<?php
/* --------------------------------------------------------------
   UrlRewrite.inc.php 2016-05-02
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2016 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * UrlRewrite
 *
 * @category System
 * @package  Shared
 */
class UrlRewrite
{
    /**
     * Holds the content type as a string.
     *
     * @var string
     */
    protected $contentType;
    
    /**
     * Holds the content ID as integer.
     *
     * @var int
     */
    protected $contentId;
    
    /**
     * Holds the language ID as integer.
     *
     * @var int
     */
    protected $languageId;
    
    /**
     * Holds the rewritten URL as string.
     *
     * @var string
     */
    protected $rewriteUrl;
    
    /**
     * Holds the original target URL as string.
     *
     * @var string
     */
    protected $targetUrl;
    
    
    /**
     * UrlRewrite constructor.
     *
     * @param NonEmptyStringType $contentType
     * @param IdType             $contentId
     * @param IdType             $languageId
     * @param NonEmptyStringType $rewriteUrl
     * @param NonEmptyStringType $targetUrl
     */
    public function __construct(
        NonEmptyStringType $contentType,
        IdType $contentId,
        IdType $languageId,
        NonEmptyStringType $rewriteUrl,
        NonEmptyStringType $targetUrl
    ) {
        $this->contentType = $contentType->asString();
        $this->contentId   = $contentId->asInt();
        $this->languageId  = $languageId->asInt();
        $this->rewriteUrl  = $this->_sanitizeUrl($rewriteUrl);
        $this->targetUrl   = $targetUrl->asString();
    }
    
    
    /**
     * @return string
     */
    public function getContentType()
    {
        return $this->contentType;
    }
    
    
    /**
     * @return int
     */
    public function getContentId()
    {
        return $this->contentId;
    }
    
    
    /**
     * @return int
     */
    public function getLanguageId()
    {
        return $this->languageId;
    }
    
    
    /**
     * @return string
     */
    public function getRewriteUrl()
    {
        return $this->rewriteUrl;
    }
    
    
    /**
     * @return string
     */
    public function getTargetUrl()
    {
        return $this->targetUrl;
    }
    
    
    /**
     * Filters a string allowing the characters a-z, 0-9, - and /.
     * Similar characters are mapped to its latin equivalent.
     *
     * @param NonEmptyStringType $rewriteUrl
     *
     * @return string
     */
    protected function _sanitizeUrl(NonEmptyStringType $rewriteUrl)
    {
        $search  = [
            'ä',
            'Ä',
            'ö',
            'Ö',
            'ü',
            'Ü',
            '&auml;',
            '&Auml;',
            '&ouml;',
            '&Ouml;',
            '&uuml;',
            '&Uuml;',
            'ß',
            '&szlig;'
        ];
        $replace = ['ae', 'Ae', 'oe', 'Oe', 'ue', 'Ue', 'ae', 'Ae', 'oe', 'Oe', 'ue', 'Ue', 'ss', 'ss'];
        $string  = str_replace($search, $replace, $rewriteUrl->asString());
        
        $search  = [
            'А',
            'Б',
            'В',
            'Г',
            'Д',
            'Е',
            'Ё',
            'Ж',
            'З',
            'И',
            'Й',
            'К',
            'Л',
            'М',
            'Н',
            'О',
            'П',
            'Р',
            'С',
            'Т',
            'У',
            'Ф',
            'Х',
            'Ц',
            'Ч',
            'Ш',
            'Щ',
            'Ъ',
            'Ы',
            'Ь',
            'Э',
            'Ю',
            'Я',
            'а',
            'б',
            'в',
            'г',
            'д',
            'е',
            'ё',
            'ж',
            'з',
            'и',
            'й',
            'к',
            'л',
            'м',
            'н',
            'о',
            'п',
            'р',
            'с',
            'т',
            'у',
            'ф',
            'х',
            'ц',
            'ч',
            'ш',
            'щ',
            'ъ',
            'ы',
            'ь',
            'э',
            'ю',
            'я'
        ];
        $replace = [
            'A',
            'B',
            'W',
            'G',
            'D',
            'Ie',
            'Io',
            'Z',
            'Z',
            'I',
            'J',
            'K',
            'L',
            'M',
            'N',
            'O',
            'P',
            'R',
            'S',
            'T',
            'U',
            'F',
            'Ch',
            'C',
            'Tch',
            'Sh',
            'Shtch',
            '',
            'Y',
            '',
            'E',
            'Iu',
            'Ia',
            'a',
            'b',
            'w',
            'g',
            'd',
            'ie',
            'io',
            'z',
            'z',
            'i',
            'j',
            'k',
            'l',
            'm',
            'n',
            'o',
            'p',
            'r',
            's',
            't',
            'u',
            'f',
            'ch',
            'c',
            'tch',
            'sh',
            'shtch',
            '',
            'y',
            '',
            'e',
            'iu',
            'ia'
        ];
        $string  = str_replace($search, $replace, $string);
        
        $search  = [
            'Á',
            'À',
            'Â',
            'Ä',
            'Ă',
            'Ā',
            'Ã',
            'Å',
            'Ą',
            'Æ',
            'Ć',
            'Ċ',
            'Ĉ',
            'Č',
            'Ç',
            'Ď',
            'Đ',
            'Ð',
            'É',
            'È',
            'Ė',
            'Ê',
            'Ë',
            'Ě',
            'Ē',
            'Ę',
            'Ə',
            'Ġ',
            'Ĝ',
            'Ğ',
            'Ģ',
            'á',
            'à',
            'â',
            'ä',
            'ă',
            'ā',
            'ã',
            'å',
            'ą',
            'æ',
            'ć',
            'ċ',
            'ĉ',
            'č',
            'ç',
            'ď',
            'đ',
            'ð',
            'é',
            'è',
            'ė',
            'ê',
            'ë',
            'ě',
            'ē',
            'ę',
            'ə',
            'ġ',
            'ĝ',
            'ğ',
            'ģ',
            'Ĥ',
            'Ħ',
            'I',
            'Í',
            'Ì',
            'İ',
            'Î',
            'Ï',
            'Ī',
            'Į',
            'Ĳ',
            'Ĵ',
            'Ķ',
            'Ļ',
            'Ł',
            'Ń',
            'Ň',
            'Ñ',
            'Ņ',
            'Ó',
            'Ò',
            'Ô',
            'Ö',
            'Õ',
            'Ő',
            'Ø',
            'Ơ',
            'Œ',
            'ĥ',
            'ħ',
            'ı',
            'í',
            'ì',
            'i',
            'î',
            'ï',
            'ī',
            'į',
            'ĳ',
            'ĵ',
            'ķ',
            'ļ',
            'ł',
            'ń',
            'ň',
            'ñ',
            'ņ',
            'ó',
            'ò',
            'ô',
            'ö',
            'õ',
            'ő',
            'ø',
            'ơ',
            'œ',
            'Ŕ',
            'Ř',
            'Ś',
            'Ŝ',
            'Š',
            'Ş',
            'Ť',
            'Ţ',
            'Þ',
            'Ú',
            'Ù',
            'Û',
            'Ü',
            'Ŭ',
            'Ū',
            'Ů',
            'Ų',
            'Ű',
            'Ư',
            'Ŵ',
            'Ý',
            'Ŷ',
            'Ÿ',
            'Ź',
            'Ż',
            'Ž',
            'ŕ',
            'ř',
            'ś',
            'ŝ',
            'š',
            'ş',
            'ß',
            'ť',
            'ţ',
            'þ',
            'ú',
            'ù',
            'û',
            'ü',
            'ŭ',
            'ū',
            'ů',
            'ų',
            'ű',
            'ư',
            'ŵ',
            'ý',
            'ŷ',
            'ÿ',
            'ź',
            'ż',
            'ž'
        ];
        $replace = [
            'A',
            'A',
            'A',
            'A',
            'A',
            'A',
            'A',
            'A',
            'A',
            'AE',
            'C',
            'C',
            'C',
            'C',
            'C',
            'D',
            'D',
            'D',
            'E',
            'E',
            'E',
            'E',
            'E',
            'E',
            'E',
            'E',
            'G',
            'G',
            'G',
            'G',
            'G',
            'a',
            'a',
            'a',
            'a',
            'a',
            'a',
            'a',
            'a',
            'a',
            'ae',
            'c',
            'c',
            'c',
            'c',
            'c',
            'd',
            'd',
            'd',
            'e',
            'e',
            'e',
            'e',
            'e',
            'e',
            'e',
            'e',
            'g',
            'g',
            'g',
            'g',
            'g',
            'H',
            'H',
            'I',
            'I',
            'I',
            'I',
            'I',
            'I',
            'I',
            'I',
            'IJ',
            'J',
            'K',
            'L',
            'L',
            'N',
            'N',
            'N',
            'N',
            'O',
            'O',
            'O',
            'O',
            'O',
            'O',
            'O',
            'O',
            'CE',
            'h',
            'h',
            'i',
            'i',
            'i',
            'i',
            'i',
            'i',
            'i',
            'i',
            'ij',
            'j',
            'k',
            'l',
            'l',
            'n',
            'n',
            'n',
            'n',
            'o',
            'o',
            'o',
            'o',
            'o',
            'o',
            'o',
            'o',
            'o',
            'R',
            'R',
            'S',
            'S',
            'S',
            'S',
            'T',
            'T',
            'T',
            'U',
            'U',
            'U',
            'U',
            'U',
            'U',
            'U',
            'U',
            'U',
            'U',
            'W',
            'Y',
            'Y',
            'Y',
            'Z',
            'Z',
            'Z',
            'r',
            'r',
            's',
            's',
            's',
            's',
            'B',
            't',
            't',
            'b',
            'u',
            'u',
            'u',
            'u',
            'u',
            'u',
            'u',
            'u',
            'u',
            'u',
            'w',
            'y',
            'y',
            'y',
            'z',
            'z',
            'z'
        ];
        $string  = str_replace($search, $replace, $string);
        
        $string = strtolower($string);
        
        $string = preg_replace('/[^a-z0-9\/]/', '-', $string);
        $string = preg_replace('/--+/', '-', $string);
        $string = preg_replace('/^-(.+)/', "$1", $string);
        $string = preg_replace('/(.+)-$/', "$1", $string);
        
        return $string;
    }
}